﻿using System;
using System.Web;
using System.Collections;
using System.Resources;
using System.Web.UI;
using System.Web.UI.HtmlControls;
using System.Web.UI.WebControls;
using System.Reflection;
using System.Collections.Generic;
using AntiXssEnc = Microsoft.Security.Application;

using VIRP_WEB.BaseCode;
using System.Linq;
using VIRP;
using VIRP.CRS.BO;
using VIRP.CRS.BLL;
using VIRP.EFR.BO;

namespace VIRP_WEB.Custom.EFR
{
    public abstract class EfrBasePage : BasePage
    {
        private const string ApplicationRole = "application";
        public string redirectAfterSave;

        protected const string DateFormatLong = "MM/dd/yyyy hh:mm:ss.fff tt";
        protected const string DateFormatShort = "M/d/yyyy";

        protected static ResourceManager resources;

        private bool errorOnSavedReceived;
        //private Dictionary<string, string> OnStartupAlertErrorCollection;

        public bool BusinessObjectSaveEventReceived { get; set; }
        public bool SuppressSaveSuccessMessage { get; set; }

        protected IReadOnlyList<string> NavigationSources { get; set; }

        protected string CurrentUser
        {
            get { return HttpContext.Current.User.Identity.Name;  }
        }

        public int CurrentRegistryId
        {
            get { return UserSession.CurrentRegistryId;  }
        }

        protected void Manager_BOSaveSuccess(object sender, EventArgs e)
        {
            if (!errorOnSavedReceived)
                BusinessObjectSaveEventReceived = true;
        }

        protected void MakeControlsReadOnly(ControlCollection controls)
        {
            foreach (Control c in controls)
            {
                if (c is TextBox || c is RadioButton ||
                    c is DropDownList || c is CheckBox || c is RadioButtonList || c is CheckBoxList)
                {
                    PropertyInfo propertyInfo = c.GetType().GetProperty("Enabled");
                    propertyInfo.SetValue(c, false, null);
                }

                if (c.HasControls())
                {
                    MakeControlsReadOnly(c.Controls);
                }
            }
        }

        protected void MakeControlsWriteable(ControlCollection controls)
        {
            foreach (Control c in controls)
            {
                if (c is TextBox || c is RadioButton ||
                    c is DropDownList || c is CheckBox || c is RadioButtonList || c is CheckBoxList)
                {
                    PropertyInfo propertyInfo = c.GetType().GetProperty("Enabled");
                    propertyInfo.SetValue(c, true, null);
                }

                if (c.HasControls())
                {
                    MakeControlsWriteable(c.Controls);
                }
            }
        }

        protected T GetOrStore<T>(string key, Func<T, bool> shouldUpdate, Func<T> getObject)
            where T : class
        {
            T result = Session[key] as T;

            if (result == default(T) || shouldUpdate(result))
            {
                result = getObject();
                Session[key] = result;
            }

            return result;
        }

        protected void SetReferringPage()
        {
            var referrer = ViewState["referrer"];
            if (referrer == null)
                ViewState["referrer"] = Request.UrlReferrer;
        }

        protected void RedirectToReferringPage()
        {
            Uri from = ViewState["referrer"] as Uri;
            if (from != null && UriUtility.IsValidPathSource(NavigationSources, from))
                Response.Redirect(from.PathAndQuery); 
        }

        /// <summary>
        /// Similar to SetReferringPage() but uses Session vs ViewState to save page originally navigated in on;
        /// useful if ViewState is not sufficient to get the original referring page
        /// </summary>
        /// <param name="pageReferrerName">ex. "BioLabInterpretation_Referrer": BioLabInterpretation save originally referring page </param>
        protected void SetOriginalReferringPage(string pageReferrerName)
        {
            var navigatedFrom = SessionItemReference<Uri>.Get(Session[pageReferrerName]);
            if (navigatedFrom == null)
                Session[pageReferrerName] = SessionItemFactory.Create(Request.UrlReferrer);
        }

        /// <summary>
        /// Similar to RedirectToReferringPage() but uses original referring page, in case page has multiple redirects;
        /// validates referrals but redirects to original referring page
        /// </summary>
        /// <param name="pageReferrerName">ex. "BioLabInterpretation_Referrer": BioLabInterpretation save originally referring page </param>
        protected void RedirectToOriginalReferringPage(string pageReferrerName)
        {
            Uri from = ViewState["referrer"] as Uri;
            if (UriUtility.IsValidPathSource(NavigationSources, from))
            {
                var navigatedFrom = SessionItemReference<Uri>.Get(Session[pageReferrerName]);
                if (navigatedFrom != null)
                {
                    Session.Remove(pageReferrerName);
                    Response.Redirect(navigatedFrom.PathAndQuery);
                }
            }
        }

        public string ConvertDateToString(DateTime? date)
        {
            if (!date.HasValue)
                return "";
            else
                return date.GetValueOrDefault().ToShortDateString();
        }

        public void PrepareGridViewForExport(Control gv)
        {
            Hashtable htControls = new Hashtable();

            htControls.Add("LinkButton", "Text");
            htControls.Add("HyperLink", "Text");
            htControls.Add("DropDownList", "SelectedItem");
            htControls.Add("CheckBox", "Checked");

            Literal l = new Literal();

            for (int i = 0; i < gv.Controls.Count; i++)
            {
                if ((null != htControls[gv.Controls[i].GetType().Name]) ||
                    (null != htControls[gv.Controls[i].GetType().BaseType.Name]))
                {
                    l.Text = GetControlPropertyValue(gv.Controls[i], htControls);
                    gv.Controls.Remove(gv.Controls[i]);
                    gv.Controls.AddAt(i, l);
                }

                if (gv.Controls[i].HasControls())
                {
                    PrepareGridViewForExport(gv.Controls[i]);
                }
            }
        }

        private string GetControlPropertyValue(Control control, Hashtable htControls)
        {
            Type controlType = control.GetType();
            string strControlType = controlType.Name;
            string strReturn = "Error";
            bool bReturn;
            PropertyInfo[] ctrlProps = controlType.GetProperties();
            string ExcelPropertyName = (string)htControls[strControlType];

            if (ExcelPropertyName == null)
            {
                ExcelPropertyName = (string)htControls[control.GetType().BaseType.Name];
                if (ExcelPropertyName == null)
                    return strReturn;
            }

            foreach (PropertyInfo ctrlProp in ctrlProps)
            {
                if (ctrlProp.Name == ExcelPropertyName && ctrlProp.PropertyType == typeof(string))
                {
                    strReturn = (string)ctrlProp.GetValue(control, null);
                    break;
                }

                if (ctrlProp.Name == ExcelPropertyName &&
                        ctrlProp.PropertyType == typeof(bool))
                {
                    bReturn = (bool)ctrlProp.GetValue(control, null);
                    strReturn = bReturn ? "True" : "False";
                    break;
                }

                if (ctrlProp.Name == ExcelPropertyName && ctrlProp.PropertyType == typeof(ListItem))
                {
                    strReturn = ((ListItem)(ctrlProp.GetValue(control, null))).Text;
                    break;
                }
            }

            return strReturn;
        }

        public void ExportGridViewToExcel(GridView GridView1, string filename)
        {
            PrepareGridViewForExport(GridView1);
            HtmlForm form = new HtmlForm();
            string attachment = "attachment; filename=" + filename;
            Response.ClearContent();
            Response.AddHeader("content-disposition", attachment);
            Response.ContentType = "application/vnd.ms-excel";
            System.IO.StringWriter sw = new System.IO.StringWriter();
            HtmlTextWriter htw = new HtmlTextWriter(sw);
            form.Controls.Add(GridView1);
            Controls.Add(form);
            form.RenderControl(htw);
            Response.Write(sw.ToString());
            Response.End();
        }

        /// <summary>
        /// Useful for Telerik WAI-ARIA (508) support
        /// </summary>
        protected void AddFormApplicationRole()
        {
            AddFormRole(ApplicationRole);
        }

        protected void AddFormRole(string role)
        {
            Form.Attributes.Add("role", role);
        }

        protected List<T> FindControls<T>(
            Control control,
            string id)
            where T : class
        {
            return control
                .GetAllControls()
                .Where(c => c is T && c.ClientID.Contains(id))
                .Select(c => c as T)
                .ToList();
        }

        protected void ReplacePageTitle()
        {
            ReplacePageTitle(Page.Title);
        }

        protected void ReplacePageTitle(string pageTitle)
        {
            ((Label)Master.FindControl("EfrPageTitle")).Text = AntiXssEnc.Encoder.HtmlEncode(pageTitle);
        }

        protected void SetSaveSuccessMessage()
        {
            SetSaveMessage("Save Successful");
        }

        protected void SetSaveMessage(string message)
        {
            ((HtmlGenericControl)Master.FindControl("lblSaveMessage")).InnerHtml = message;

            ClientScript.RegisterStartupScript(this.GetType(), "scrollTop", "<script type='text/javascript'>$('html, body').animate({ scrollTop: 0 }, 'slow');</script>");
        }

        protected string GetShortDateString(object dateObject)
        {
            return GetDateString(dateObject, DateFormatShort);
        }

        protected string GetLongDateString(object dateObject)
        {
            return GetDateString(dateObject, DateFormatLong);
        }

        private static string GetDateString(object dateObject, string format)
        {
            if (dateObject == null) return string.Empty;

            string dateString = Convert.ToString(dateObject);

            DateTime date;
            if (DateTime.TryParse(dateString, out date))
                return date.ToString(format);

            return string.Empty;
        }

        protected PatientSummary GetOrStorePatientSummary()
        {
            int patientId = 0;
            int.TryParse(Request.QueryString["patientId"], out patientId);

            return patientId > 0 ?
                GetOrStore("PatientSummary",
                    (existing) =>
                    {
                        return existing.Id != patientId;
                    },
                    () =>
                    {
                        PATIENT patient = PATIENTManager.GetItem(HttpContext.Current.User.Identity.Name, UserSession.CurrentRegistryId, patientId);
                        PATIENT_REG patientReg = PATIENT_REGManager.GetItem(HttpContext.Current.User.Identity.Name, UserSession.CurrentRegistryId, patientId);
                        return new PatientSummary(patient, patientReg);
                    })
                : null;
        }
    }
}